// Copyright 2020 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/cr_view_manager/cr_view_manager.m.js';
import 'chrome://resources/cr_elements/cr_lazy_render/cr_lazy_render.m.js';
import './profile_picker_main_view.js';
import './profile_picker_shared_css.js';

import {assert, assertNotReached} from 'chrome://resources/js/assert.m.js';
import {loadTimeData} from 'chrome://resources/js/load_time_data.m.js';
import {html, Polymer} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {ensureLazyLoaded} from './ensure_lazy_loaded.js';
import {AutogeneratedThemeColorInfo, ManageProfilesBrowserProxy, ManageProfilesBrowserProxyImpl} from './manage_profiles_browser_proxy.js';
import {navigateTo, NavigationBehavior, ProfileCreationSteps, Routes} from './navigation_behavior.js';
import {isProfileCreationAllowed} from './policy_helper.js';

Polymer({
  is: 'profile-picker-app',

  _template: html`{__html_template__}`,

  behaviors: [NavigationBehavior],

  properties: {
    /**
     * Suggested new profile theme info for the profile creation flow.
     * @type {!AutogeneratedThemeColorInfo}
     */
    newProfileThemeInfo: {
      type: Object,
      notify: true,
    },
  },

  /** @private {?Routes} */
  currentRoute_: null,

  /** @private {?ManageProfilesBrowserProxy} */
  manageProfilesBrowserProxy_: null,

  /** @override */
  ready() {
    this.manageProfilesBrowserProxy_ =
        ManageProfilesBrowserProxyImpl.getInstance();
  },

  /** @override */
  attached() {
    this.setMinimumSize_();
  },

  /**
   * @param {Routes} route
   * @param {string} step
   * @private
   */
  onRouteChange(route, step) {
    if (!isProfileCreationAllowed() && route === Routes.NEW_PROFILE) {
      navigateTo(Routes.MAIN);
      return;
    }

    if (step == ProfileCreationSteps.LOAD_SIGNIN) {
      this.manageProfilesBrowserProxy_.loadSignInProfileCreationFlow(
          this.newProfileThemeInfo.color);
      return;
    }

    const setStep = () => {
      this.$.viewManager.switchView(step, 'fade-in', 'no-animation');
    };

    // If the route changed, initialize modules for that route.
    if (this.currentRoute_ !== route) {
      this.currentRoute_ = route;
      this.initializeModules_().then(setStep);
    } else {
      setStep();
    }
  },

  /**
   * @return {!Promise}
   * @private
   */
  initializeModules_() {
    switch (this.currentRoute_) {
      case Routes.MAIN:
        return Promise.resolve();
      case Routes.NEW_PROFILE:
        return Promise.all(
            [this.initializeNewProfileThemeInfo_(), ensureLazyLoaded()]);
      default:
        // |this.currentRoute_| should be set by now.
        assertNotReached();
    }
  },

  /**
   * @return {!Promise}
   * @private
   */
  async initializeNewProfileThemeInfo_() {
    this.newProfileThemeInfo = await this.manageProfilesBrowserProxy_
                                   .getNewProfileSuggestedThemeInfo();
  },

  /** @private */
  setMinimumSize_() {
    this.style.setProperty(
        '--view-min-size', loadTimeData.getString('minimumPickerSize'));
  },
});
